<?php

namespace App\Notifications;

use App\Models\Doctor;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class DoctorRegistrationNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $doctor;

    /**
     * Create a new notification instance.
     */
    public function __construct(Doctor $doctor)
    {
        $this->doctor = $doctor;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        $channels = ['database'];
        
        // Include mail channel - errors will be caught by try-catch in controllers
        // For development, set MAIL_MAILER=log in .env to avoid connection errors
        $channels[] = 'mail';
        
        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('New Doctor Registration - ' . config('app.name'))
            ->greeting('Hello Admin,')
            ->line('A new doctor has registered and requires verification.')
            ->line('**Doctor Name:** ' . $this->doctor->full_name)
            ->line('**Email:** ' . $this->doctor->email)
            ->line('**Practice Number:** ' . $this->doctor->practice_number)
            ->line('**Specialization:** ' . ($this->doctor->specialization ?? 'N/A'))
            ->action('Review Doctor', route('admin.doctors'))
            ->line('Thank you for using ' . config('app.name') . '!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'doctor_id' => $this->doctor->id,
            'doctor_name' => $this->doctor->full_name,
            'practice_number' => $this->doctor->practice_number,
            'message' => 'New doctor registration: ' . $this->doctor->full_name . ' requires verification',
            'type' => 'doctor_registration',
        ];
    }
}
